using System;
using System.Runtime.InteropServices;

namespace Sony
{
	namespace NP
	{
		public class WordFilter
		{
			// Structure for returning the word filter result.
			[StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi, Pack = 0)]
			public struct FilteredComment
			{
				public bool wasChanged;
				IntPtr _comment;
				public string comment { get { return Marshal.PtrToStringAnsi(_comment); } }
			};

			[DllImport("UnityNpToolkit")]
			private static extern bool PrxWordFilterGetLastError(out ResultCode result);
			
			public static bool GetLastError(out ResultCode result)
			{
				PrxWordFilterGetLastError(out result);
				return result.lastError == ErrorCode.NP_OK;
			}

			[DllImport("UnityNpToolkit")]
			private static extern bool PrxWordFilterIsBusy();
			[DllImport("UnityNpToolkit", CharSet = CharSet.Ansi)]
			private static extern bool PrxWordFilterCensorComment(string name);
			[DllImport("UnityNpToolkit", CharSet = CharSet.Ansi)]
			private static extern bool PrxWordFilterSanitizeComment(string name);
			[DllImport("UnityNpToolkit")]
			private static extern bool PrxWordFilterGetResult(out FilteredComment result);

			// Is the word filter busy with a request.
			public static bool IsBusy()
			{
				return PrxWordFilterIsBusy();
			}

			// Censor a comment, handle results by adding event handlers for the OnCommentCensored & OnCommentNotCensored events and calling GetResult() to get the modified comment.
			public static bool CensorComment(string comment)
			{
				return PrxWordFilterCensorComment(comment);
			}

			// Sanitize a comment, handle results by adding an event handler for the OnCommentSanitized event and calling GetResult() to get the modified comment.
			public static bool SanitizeComment(string comment)
			{
				return PrxWordFilterSanitizeComment(comment);
			}

			// Get the result of the word filter request.
			public static FilteredComment GetResult()
			{
				FilteredComment result = new FilteredComment();
				PrxWordFilterGetResult(out result);
				return result;
			}

			// Event delegates.
			public static event Messages.EventHandler OnCommentNotCensored;
			public static event Messages.EventHandler OnCommentCensored;
			public static event Messages.EventHandler OnCommentSanitized;
			public static event Messages.EventHandler OnWordFilterError;

			// Message processing.
			public static bool ProcessMessage(Messages.PluginMessage msg)
			{
				switch (msg.type)
				{
					case Messages.MessageType.kNPToolKit_WordFilterNotCensored:
						if (OnCommentNotCensored != null) OnCommentNotCensored(msg);
						return true;
						
					case Messages.MessageType.kNPToolKit_WordFilterCensored:
						if (OnCommentCensored != null) OnCommentCensored(msg);
						return true;
						
					case Messages.MessageType.kNPToolKit_WordFilterSanitized:
						if (OnCommentSanitized != null) OnCommentSanitized(msg);
						return true;

					case Messages.MessageType.kNPToolKit_WordFilterError:
						if (OnWordFilterError != null) OnWordFilterError(msg);
						return true;
				}
				
				return false;
			}

		}
	}
}
